MACRO (SKIP_AWS_SDK MSG)
  SET(${RETVAL} OFF PARENT_SCOPE)
  SET(${REASON} ${MSG} PARENT_SCOPE)
  RETURN()
ENDMACRO()

FUNCTION (CHECK_AWS_SDK COMPONENTS RETVAL REASON)
  # AWS_SDK_EXTERNAL_PROJECT must be ON
  IF(NOT AWS_SDK_EXTERNAL_PROJECT)
   FOREACH(comp ${COMPONENTS})
    FIND_PACKAGE(aws-cpp-sdk-${comp} CONFIG QUIET)
    IF (NOT aws-cpp-sdk-${comp}_FOUND)
      SKIP_AWS_SDK("AWS_SDK_EXTERNAL_PROJECT is not ON and aws-cpp-sdk-${comp} not found")
    ENDIF()
   ENDFOREACH()
   SET(${RETVAL} ON PARENT_SCOPE)
  ENDIF()
  IF(NOT NOT_FOR_DISTRIBUTION)
    SKIP_AWS_SDK("AWS SDK has Apache 2.0 License which is not compatible with GPLv2. Set -DNOT_FOR_DISTRIBUTION=ON if you need it")
  ENDIF()
  IF(CMAKE_VERSION VERSION_LESS "3.15")
    SKIP_AWS_SDK("CMake too old")
  ENDIF()
  # Check compiler support
  IF(CMAKE_CXX_COMPILER_ID MATCHES "GNU")
    IF (CMAKE_CXX_COMPILER_VERSION VERSION_LESS 4.9)
      SKIP_AWS_SDK("GCC VERSION too old (${GCC_VERSION}, required is 4.9 or later")
    ENDIF()
  ELSEIF (CMAKE_CXX_COMPILER_ID MATCHES "Clang")
    IF ((CMAKE_CXX_COMPILER_VERSION AND CMAKE_CXX_COMPILER_VERSION VERSION_LESS  3.3) OR
        (CLANG_VERSION_STRING AND CLANG_VERSION_STRING VERSION_LESS  3.3))
      SKIP_AWS_SDK("Clang version too old, required is 3.3 or later")
    ENDIF()
  ELSEIF(MSVC)
    IF (MSVC_VERSION LESS 1800)
      SKIP_AWS_SDK("MSVC version too old, required is VS2015 or later")
    ENDIF()
  ELSE()
    SKIP_AWS_SDK("Unsupported compiler")
  ENDIF()

  # Check OS support
  IF (NOT(WIN32 OR APPLE  OR (CMAKE_SYSTEM_NAME MATCHES "Linux")))
    SKIP_AWS_SDK("OS unsupported by AWS SDK")
  ENDIF()

  IF(UNIX)
    IF("${WITH_ZLIB}" STREQUAL "bundled")
      # Breaks FIND_PACKAGE(ZLIB)
      SKIP_AWS_SDK("Incompatible with WITH_ZLIB=bundled")
    ENDIF()
    # Check libraries required for building SDK
    FOREACH(pkg CURL ZLIB OpenSSL)
      FIND_PACKAGE(${pkg})
      IF(NOT ${pkg}_FOUND)
        SKIP_AWS_SDK("AWS C++ SDK requires ${pkg}")
      ENDIF()
      SET_PACKAGE_PROPERTIES(${pkg} PROPERTIES TYPE REQUIRED)
    ENDFOREACH()
    # Also check for required libraries explicitely - they might be
    # missing, even if check above succeeds, e.g when using own copy
    # of zlib
    FOREACH(lib OpenSSL::Crypto ZLIB::ZLIB CURL::libcurl)
      IF(NOT TARGET ${lib})
        SKIP_AWS_SDK("AWS C++ SDK requires ${lib}")
      ENDIF()
    ENDFOREACH()
  ENDIF()
  SET(${RETVAL} ON PARENT_SCOPE)
ENDFUNCTION()


# USE_AWS_SDK_LIBS(target sdk_component1 ... sdk_component_N)
# Example usage
# USE_AWS_SDK_LIBS(aws_key_management kms s3)
FUNCTION(USE_AWS_SDK_LIBS)
  SET(SDK_COMPONENTS ${ARGN})
  LIST(GET SDK_COMPONENTS 0 target)
  IF(NOT TARGET ${target})
    MESSAGE(FATAL_ERROR "${target} is not a valid target")
  ENDIF()
  SET(NON_DISTRIBUTABLE_WARNING "Apache 2.0" CACHE INTERNAL "")
  LIST(REMOVE_AT SDK_COMPONENTS 0)
  FOREACH(comp ${SDK_COMPONENTS})
    SET_PROPERTY(GLOBAL PROPERTY AWS_SDK_LIBS ${comp} APPEND)
    TARGET_LINK_LIBRARIES(${target} aws-cpp-sdk-${comp})
  ENDFOREACH()
ENDFUNCTION()
